package ga.view.billard;

import ga.core.algorithm.interactive.ISIGA;
import ga.core.algorithm.util.RandomSingleton;
import ga.core.evaluation.EvaluationListener;
import ga.core.evaluation.IInteractiveFitnessEvaluator;
import ga.core.individual.IAgeIndividual;
import ga.core.individual.IIndividualFactory;
import ga.view.appstate.SceneState;
import ga.view.appstate.menu.IMenuListenerParent;
import ga.view.appstate.menu.MenuListener;
import ga.view.billard.nodes.BillardObjectNode;
import ga.view.factory.EffectsFactory;
import ga.view.interfaces.IPhenotypeGenerator;
import ga.view.interfaces.MouseListener;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;

import com.jme3.app.Application;
import com.jme3.app.state.AppStateManager;
import com.jme3.bullet.BulletAppState;
import com.jme3.bullet.PhysicsSpace;
import com.jme3.bullet.collision.shapes.BoxCollisionShape;
import com.jme3.bullet.control.RigidBodyControl;
import com.jme3.collision.CollisionResult;
import com.jme3.collision.CollisionResults;
import com.jme3.input.KeyInput;
import com.jme3.input.controls.ActionListener;
import com.jme3.input.controls.KeyTrigger;
import com.jme3.input.controls.MouseButtonTrigger;
import com.jme3.light.AmbientLight;
import com.jme3.light.DirectionalLight;
import com.jme3.material.Material;
import com.jme3.math.ColorRGBA;
import com.jme3.math.Ray;
import com.jme3.math.Vector3f;
import com.jme3.renderer.queue.RenderQueue;
import com.jme3.scene.Geometry;
import com.jme3.scene.Node;
import com.jme3.scene.Spatial;
import com.jme3.scene.shape.Box;
import com.jme3.system.AppSettings;

/**
 * This interactive evaluator is a special case for direct selection.
 * 
 * @param <T>
 *          The generic type of the individuals.
 * 
 * @since 12.08.2012
 * @author Stephan Dreyer
 */
@SuppressWarnings("unchecked")
public class BillardEvaluationState<T extends IAgeIndividual<T>> extends
    SceneState implements IInteractiveFitnessEvaluator<T>, IMenuListenerParent {
  private static final Logger LOGGER = Logger
      .getLogger(BillardEvaluationState.class.getSimpleName());

  private Node objectsNode;

  private final BulletAppState bulletAppState;

  private ClickListener dragListener;

  private final float scale = 1.5f;
  private final float sceneHeight = 5f * scale;
  private float sceneWidth;
  private final float wallHeight = 2f * scale;
  private final float wallThickness = .1f;

  private ISIGA<T> algorithm;
  private final IPhenotypeGenerator<T, Geometry> phenotypeGenerator;
  private final IIndividualFactory<T> factory;

  private MenuListener menuListener;
  private final List<EvaluationListener<T>> listeners = new ArrayList<EvaluationListener<T>>();

  private final Map<T, BillardObjectNode<T>> nodeMap = new HashMap<T, BillardObjectNode<T>>();

  private boolean paused;
  private boolean useCentering;
  private boolean debug;

  /**
   * Instantiates a new billard evaluation state.
   * 
   * @param factory
   *          the factory
   * @param phenotypeGenerator
   *          the phenotype generator
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public BillardEvaluationState(final IIndividualFactory<T> factory,
      final IPhenotypeGenerator<T, Geometry> phenotypeGenerator) {
    this.factory = factory;
    this.phenotypeGenerator = phenotypeGenerator;
    bulletAppState = new BulletAppState();
    bulletAppState.setThreadingType(BulletAppState.ThreadingType.PARALLEL);
  }

  @Override
  public void initialize(final AppStateManager stateManager,
      final Application app) {
    super.initialize(stateManager, app);

    phenotypeGenerator.setAssetManager(assetManager);

    final AppSettings settings = app.getContext().getSettings();
    final float aspectRatio = settings.getWidth()
        / (float) settings.getHeight();
    sceneWidth = sceneHeight * aspectRatio;

    stateManager.attach(bulletAppState);
    bulletAppState.getPhysicsSpace().setAccuracy(0.003f);

    objectsNode = new Node("Panels");
    rootNode.attachChild(objectsNode);

    final DirectionalLight dl = new DirectionalLight();
    dl.setDirection(new Vector3f(0f, -1f, 0f));
    rootNode.addLight(dl);

    rootNode.addLight(new AmbientLight());

    initCam();
    initTable();
    initObjects();

    // add the drag mapping
    dragListener = new ClickListener();
    inputManager.addMapping("drag", new MouseButtonTrigger(0));
    inputManager.addListener(dragListener, "drag");

    inputManager.addMapping("paused", new KeyTrigger(KeyInput.KEY_SPACE));
    inputManager.addListener(new PausedListener(), "paused");
    inputManager.addMapping("debug", new KeyTrigger(KeyInput.KEY_D));
    inputManager.addListener(new DebugListener(), "debug");
    inputManager.addMapping("centering", new KeyTrigger(KeyInput.KEY_C));
    inputManager.addListener(new CenteringListener(), "centering");

    EffectsFactory.addShadowProcessor(assetManager, settings, viewPort,
        new Vector3f(0f, 1f, 0.1f));
    EffectsFactory.addLightScatteringProcessor(assetManager, inputManager,
        settings, viewPort, Vector3f.ZERO);
    EffectsFactory.addSSAOProcessor(assetManager, inputManager, settings,
        viewPort);
  }

  @Override
  public void cleanup() {
    stateManager.detach(bulletAppState);

    super.cleanup();
  }

  @Override
  public void setEnabled(final boolean enabled) {
    super.setEnabled(enabled);

    bulletAppState.setEnabled(enabled);

    if (dragListener != null) {
      dragListener.setEnabled(enabled);
    }

    if (viewPort != null) {
      if (enabled) {
        if (!renderManager.getMainViews().contains(viewPort)) {
          viewPort = renderManager.createMainView("Scene", cam);
          viewPort.setClearFlags(true, true, true);
          viewPort.attachScene(rootNode);
        }
      } else {
        renderManager.removeMainView(viewPort);
      }
    }
  }

  @Override
  public void update(final float tpf) {
    final Iterator<Spatial> it = objectsNode.getChildren().iterator();
    while (it.hasNext()) {
      final Spatial s = it.next();
      if (s instanceof BillardObjectNode) {
        if (((BillardObjectNode<T>) s).mustRemove()) {
          it.remove();
        }
      }

    }

    super.update(paused ? 0 : tpf);
  }

  /**
   * Pauses the rendering and physics calculation.
   * 
   * @param paused
   *          the new paused flag.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public void setPaused(final boolean paused) {
    this.paused = paused;
    bulletAppState.setEnabled(!paused);
  }

  @Override
  public void setAlgorithm(final ISIGA<T> siga) {
    this.algorithm = siga;
  }

  @Override
  public ISIGA<T> getAlgorithm() {
    return algorithm;
  }

  /**
   * Inits the camera.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private void initCam() {
    /** Set up camera */
    cam.setLocation(new Vector3f(0f, 13f * scale, 0f));
    cam.lookAt(Vector3f.ZERO, Vector3f.UNIT_Z.negate());
    cam.setFrustumFar(100f);

  }

  /** Make a solid floor and add it to the scene. */
  private void initTable() {
    final Material mat = new Material(assetManager,
        "Common/MatDefs/Light/Lighting.j3md");
    mat.setFloat("Shininess", 0.1f);
    mat.setBoolean("UseMaterialColors", true);
    mat.setBoolean("UseAlpha", false);
    mat.setColor("Ambient", ColorRGBA.Black);
    mat.setColor("Diffuse", ColorRGBA.LightGray);
    mat.setColor("Specular", ColorRGBA.Black);

    final Material planeMat = new Material(assetManager,
        "Common/MatDefs/Light/Lighting.j3md");
    planeMat.setFloat("Shininess", 0.0f);
    planeMat.setBoolean("UseMaterialColors", true);
    planeMat.setBoolean("UseAlpha", false);
    planeMat.setColor("Ambient", ColorRGBA.Black);
    planeMat.setColor("Diffuse", ColorRGBA.LightGray);
    planeMat.setColor("Specular", ColorRGBA.Black);

    Vector3f halfExtends = new Vector3f(sceneWidth, wallHeight, wallThickness);

    Geometry geo = new Geometry("upperbox", new Box(halfExtends.x,
        halfExtends.y, halfExtends.z));
    geo.setMaterial(mat);
    geo.setLocalTranslation(0, 0f, -sceneHeight);

    RigidBodyControl control = new RigidBodyControl(new BoxCollisionShape(
        halfExtends.clone()), 0);
    geo.addControl(control);
    rootNode.attachChild(geo);
    getPhysicsSpace().add(control);

    geo = new Geometry("lowerbox", new Box(halfExtends.x, halfExtends.y,
        halfExtends.z));
    geo.setLocalTranslation(0, 0f, sceneHeight);
    geo.setMaterial(mat);

    control = new RigidBodyControl(new BoxCollisionShape(halfExtends), 0);
    geo.addControl(control);

    rootNode.attachChild(geo);
    getPhysicsSpace().add(control);

    halfExtends = new Vector3f(wallThickness, wallHeight, sceneHeight);

    geo = new Geometry("leftbox", new Box(halfExtends.x, halfExtends.y,
        halfExtends.z));
    geo.setLocalTranslation(sceneWidth, 0f, 0);
    geo.setMaterial(mat);

    control = new RigidBodyControl(new BoxCollisionShape(halfExtends), 0);
    geo.addControl(control);

    rootNode.attachChild(geo);
    getPhysicsSpace().add(control);

    geo = new Geometry("rightbox", new Box(halfExtends.x, halfExtends.y,
        halfExtends.z));
    geo.setLocalTranslation(-sceneWidth, 0f, 0);
    geo.setMaterial(mat);

    control = new RigidBodyControl(new BoxCollisionShape(halfExtends), 0);
    geo.addControl(control);

    rootNode.attachChild(geo);
    getPhysicsSpace().add(control);

    halfExtends = new Vector3f(sceneWidth, wallThickness, sceneHeight);

    geo = new Geometry("plane", new Box(halfExtends.x, halfExtends.y,
        halfExtends.z));
    geo.setMaterial(planeMat);

    control = new RigidBodyControl(new BoxCollisionShape(halfExtends), 0);
    geo.addControl(control);
    geo.setShadowMode(RenderQueue.ShadowMode.Receive);

    // set a high friction to stop objects
    // control.setFriction(1f);
    rootNode.attachChild(geo);
    getPhysicsSpace().add(control);

    final Node n = new Node("upper plane node");
    n.setLocalTranslation(0, wallHeight * .9f, 0);

    control = new RigidBodyControl(new BoxCollisionShape(halfExtends), 0);
    n.addControl(control);

    // node.attachDebugShape(assetManager);
    rootNode.attachChild(n);
    getPhysicsSpace().add(control);
  }

  /**
   * Inits the objects on the table.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private void initObjects() {
    int count = 0;
    for (float x = -sceneWidth * .7f; x < sceneWidth - 1f; x += sceneHeight / 2f) {
      for (float y = -sceneHeight * .7f; y < sceneHeight; y += sceneHeight / 2f) {
        count++;
        addRandomObject(x, y);
      }
    }

    for (final T soi : algorithm.getPopulation()) {
      soi.setMaxAge(count);
    }
  }

  /**
   * Creates a random object and places it at x,y on the table.
   * 
   * @param x
   *          The x location.
   * @param z
   *          The y location.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private void addRandomObject(final float x, final float z) {
    final T newInd = factory.newIndividual(null);
    final Geometry geo = phenotypeGenerator.createPhenotype(newInd);

    final BillardObjectNode<T> newNode = new BillardObjectNode<T>(assetManager,
        geo, newInd);
    newNode.setLocalTranslation(x, 2, z);

    algorithm.getPopulation().addIndividual(newInd);

    objectsNode.attachChild(newNode);
    getPhysicsSpace().add(newNode);
    nodeMap.put(newInd, newNode);
  }

  /**
   * Finds spatials in the scene that has been clicked.
   * 
   * @param node
   *          The parent node to check for clicks.
   * @return The results of the click.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private CollisionResults findPick(final Node node) {
    node.updateGeometricState();
    final Vector3f origin = cam.getWorldCoordinates(
        inputManager.getCursorPosition(), 0.0f);
    final Vector3f direction = cam.getWorldCoordinates(
        inputManager.getCursorPosition(), 0.3f);
    direction.subtractLocal(origin).normalizeLocal();

    final Ray ray = new Ray(origin, direction);
    final CollisionResults results = new CollisionResults();
    node.collideWith(ray, results);
    return results;
  }

  /**
   * Getter for the physics space.
   * 
   * @return The physics space.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private PhysicsSpace getPhysicsSpace() {
    return bulletAppState.getPhysicsSpace();
  }

  @Override
  public void fireNewIndividualRequested() {
    for (final EvaluationListener<T> l : listeners) {
      l.newIndividualRequested();
    }
  }

  @Override
  public void fireIndividualEvaluated(final T individual) {
    for (final EvaluationListener<T> l : listeners) {
      l.individualEvaluated(individual);
    }
  }

  @Override
  public void addEvaluationListener(final EvaluationListener<T> listener) {
    listeners.add(listener);
  }

  @Override
  public void removeEvaluationListener(final EvaluationListener<T> listener) {
    listeners.remove(listener);
  }

  @Override
  public void evaluate(final T individual) {
    // do nothing
  }

  @Override
  public void setMenuListener(final MenuListener menuListener) {
    this.menuListener = menuListener;
  }

  /**
   * Mouse listener to recognize clicks on objects.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private class ClickListener extends MouseListener {

    @Override
    public void onAction(final String name, final boolean keyPressed,
        final boolean isDoubleClick, final float tpf) {

      if (keyPressed) {
        if (paused) {
          setPaused(false);
        } else {
          final CollisionResults results = findPick(objectsNode);

          if (results.size() > 0) {
            final CollisionResult closest = results.getClosestCollision();
            final Node n = closest.getGeometry().getParent();
            if (n instanceof BillardObjectNode) {
              final BillardObjectNode<T> node1 = (BillardObjectNode<T>) n;

              LOGGER.info("Objects count: " + objectsNode.getChildren().size());

              BillardObjectNode<T> node2;
              do {
                final int i = RandomSingleton.getRandom().nextInt(
                    objectsNode.getChildren().size());
                node2 = (BillardObjectNode<T>) objectsNode.getChild(i);
              } while (node2 == n);

              final T ind1 = node1.getIndividual();
              final T ind2 = node2.getIndividual();

              T newInd = algorithm.getCrossoverOp()
                  .crossover(ind1, ind2, algorithm.getContext())
                  .get(RandomSingleton.getRandom().nextInt(2));

              newInd = algorithm.getMutationOp().mutate(newInd,
                  algorithm.getContext());

              final Geometry geo = phenotypeGenerator.createPhenotype(newInd);

              final BillardObjectNode<T> newNode = new BillardObjectNode<T>(
                  assetManager, geo, newInd);

              Vector3f loc;

              // TODO THIS IS THE LOCATION CALC
              if (useCentering) {
                loc = node1.getLocalTranslation()
                    .add(node2.getLocalTranslation()).mult(.5f);
              } else {
                loc = node1.getLocalTranslation().add(
                    node1.getLocalTranslation().negate().normalize());
              }

              newNode.setLocalTranslation(loc);

              node1.doSelect();
              node2.doSelect();

              getPhysicsSpace().add(newNode);
              objectsNode.attachChild(newNode);
              algorithm.getPopulation().addIndividual(newInd);
              nodeMap.put(newInd, newNode);

              newNode.doFadeIn();

              final int maxAge = algorithm.getPopulation().size();

              // /// TODO THIS IS A QUICK-AND-DIRTY HACK AND NO GOOD PRACTICE
              final Iterator<T> it = algorithm.getPopulation().iterator();
              while (it.hasNext()) {
                final T soi = it.next();

                soi.setMaxAge(maxAge);
                soi.incAge();

                final BillardObjectNode<T> son = nodeMap.get(soi);
                son.updateSizeByAge();
                // getPhysicsSpace().remove(son);
                // getPhysicsSpace().add(son);

                if (soi.isOld()) {
                  it.remove();
                  nodeMap.remove(soi);

                  getPhysicsSpace().remove(son);
                  // objectsNode.detachChild(son);
                  son.doFadeOut();
                }
              }
              // //// HACK END

              // pause for debug
              if (debug) {
                setPaused(true);
              }
            }
          }
        }
      }
    }
  }

  /**
   * Key listener to toggle debug display.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private class DebugListener implements ActionListener {
    @Override
    public void onAction(final String name, final boolean keyPressed,
        final float tpf) {
      if (!keyPressed) {
        debug = !debug;
        LOGGER.log(Level.INFO, "debug enabled: " + debug);
      }
    }
  }

  /**
   * Key listener to toggle pause.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private class PausedListener implements ActionListener {
    @Override
    public void onAction(final String name, final boolean keyPressed,
        final float tpf) {
      if (!keyPressed) {
        setPaused(!paused);
        LOGGER.log(Level.INFO, "paused: " + paused);
      }
    }
  }

  /**
   * Key listener to toggle centering behavior. Centering means a new object
   * will created between the two parents.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private class CenteringListener implements ActionListener {
    @Override
    public void onAction(final String name, final boolean keyPressed,
        final float tpf) {
      if (!keyPressed) {
        useCentering = !useCentering;
        LOGGER.log(Level.INFO, "use centering: " + useCentering);
      }
    }
  }

}
